# MOZRobot SDK Documentation

[![Python 3.10+](https://img.shields.io/badge/python-3.10+-blue.svg)](https://www.python.org/downloads/)
[![ROS2 Humble](https://img.shields.io/badge/ROS2-Humble-brightgreen.svg)](https://docs.ros.org/en/humble/)

## Overview

MOZRobot SDK is a comprehensive Python SDK for controlling MOZ robot systems. Built on ROS2, it provides intuitive APIs for robot control and data collection for the Moz1 robot.

### Key Features

- **Multi-configuration Support**: Dual-arm, full-body (with/without base)
- **Real-time Control**: High-frequency control loops (up to 120Hz) for precise manipulation
- **Vision Integration**: Native RealSense camera support with multi-camera configuration
- **ROS2 Native**: Seamless integration with ROS2 Humble ecosystem
- **Safety First**: Built-in safety checks and error handling
- **Easy Integration**: Simple Python API for rapid prototyping and deployment

## Quick Start

### Performance Optimization Configuration (Optional)

MOZRobot SDK supports real-time performance optimization for scenarios that require strict control precision and low latency:

#### Soft Real-time Scheduling
Enabling soft real-time scheduling improves the real-time performance of robot control, reducing control latency and jitter.

```bash
# Run setup script (requires administrator privileges)
sudo bash scripts/setup_rtprio.sh
sudo reboot
```

#### CPU Binding
Bind the robot control thread to specific CPU cores to avoid task switching overhead and improve control stability.

**Recommended Configuration**:
- Avoid using core 0, typically occupied by system tasks

**Important Notes**:
- These features are optional and not required for normal use
- Enable only when optimal real-time performance is needed
- Ensure sufficient CPU resources are available for binding

### System Requirements

- **Operating System**: Ubuntu 22.04 LTS (recommended)
- **ROS Version**: ROS2 Humble
- **Python Version**: 3.10 or higher, system interpreter recommended
- **Hardware Requirements**:
  - RealSense cameras: USB 3.0 × 3
  - Robot control system connection: RJ45 × 1
- **System Resources**: 8GB+ RAM recommended

### Supported Platforms

- x86_64
- aarch64

### Network Configuration

- **Host IP**: Must be in 172.16.0.x subnet (e.g., 172.16.0.100)
- **Robot IP**: 172.16.0.20
- **ROS_DOMAIN_ID**: 33 (consistent with MovaX control system)

## Environment Setup Reference

### Step 1: System Dependencies

[ROS Humble Installation Guide](https://docs.ros.org/en/humble/Installation/Ubuntu-Install-Debs.html)

```bash
# Install ROS2 Humble
sudo apt install software-properties-common
sudo add-apt-repository universe
sudo apt update && sudo apt install curl -y

sudo curl -sSL https://raw.githubusercontent.com/ros/rosdistro/master/ros.key -o /usr/share/keyrings/ros-archive-keyring.gpg
echo "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/ros-archive-keyring.gpg] http://packages.ros.org/ros2/ubuntu $(. /etc/os-release && echo $UBUNTU_CODENAME) main" | sudo tee /etc/apt/sources.list.d/ros2.list > /dev/null

sudo apt update
# you can install ros-humble-desktop-full
sudo apt install -y ros-humble-ros-base python3-colcon-common-extensions ros-dev-tools

# Install system dependencies
sudo apt install -y build-essential cmake git curl wget python3-pip python3-dev
sudo apt install -y libopencv-dev python3-opencv
```

### Step 2: robot_interface ROS Package Build

The robot_interface is a required ROS2 interface package containing robot communication messages and service definitions. This package must be built before using the SDK.

```bash
# Navigate to dependencies directory
cd /path/to/mozrobot-SDK/deps

# Create ROS2 workspace and build robot_interface package
mkdir -p ~/ros_ws/src
cp -r robot_interface ~/ros_ws/src/
cd ~/ros_ws

# Configure ROS2 environment
source /opt/ros/humble/setup.bash
export ROS_DOMAIN_ID=33

# Build package
colcon build --packages-select mc_core_interface

# Set package environment variables (Important: run before each use)
source install/setup.bash
```

### Step 3: Environment Configuration

```bash
# Configure ROS2 environment
echo "source /opt/ros/humble/setup.bash" >> ~/.bashrc
echo "export ROS_DOMAIN_ID=33" >> ~/.bashrc
echo "source ~/mozrobot/deps/ros_ws/install/setup.bash" >> ~/.bashrc  # Add robot_interface package environment
source ~/.bashrc
```

### Step 4: Network Configuration

Edit network configuration:

```bash
sudo nano /etc/netplan/01-netcfg.yaml
```

Add the following configuration:

```yaml
network:
  version: 2
  ethernets:
    eth0:  # Replace with actual network interface name
      addresses: [172.16.0.100/24]  # Choose IP in 172.16.0.x range
      gateway4: 172.16.0.1
      nameservers:
        addresses: [8.8.8.8, 8.8.4.4]
```

Apply configuration:

```bash
sudo netplan apply
```

### Step 5: SDK Installation

```bash
# Install Python dependencies
cd /path/to/mozrobot-sdk
pip3 install -r packages/requirements.txt

# Install SDK
pip install packages/mozrobot-*.whl
```

### Step 6: Verify Installation

```bash
python3 -c "from mozrobot import MOZ1Robot, MOZ1RobotConfig; print('✅ MOZRobot SDK installed successfully!')"
```

### Quick Test

```bash
# Set environment
export ROS_DOMAIN_ID=33

# Run example
cd mozrobot
python3 example/example_usage.py

# Enable performance optimization example
python3 example/example_usage.py --enable-soft-realtime --bind-cpu 5 --no-camera
```

## API Reference

### Robot Configuration

#### `MOZ1RobotConfig`

Main configuration class for setting robot parameters.

**Constructor Parameters:**

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `realsense_serials` | `str` | `"0000000000,0000000000,0000000000"` | RealSense camera serial numbers (comma-separated) |
| `camera_resolutions` | `str` | `"320*240,320*240,320*240"` | Camera resolutions in "width*height" format |
| `no_camera` | `bool` | `False` | Skip camera initialization (for testing) |
| `structure` | `str` | `"wholebody"` | Robot configuration type |
| `robot_control_hz` | `int` | `120` | Control loop frequency (Hz), consistent with robot control system settings |
| `enable_soft_realtime` | `bool` | `False` | Enable soft real-time scheduling, requires running `scripts/setup_rtprio.sh` first |
| `bind_cpu_idxs` | `Optional[Sequence[int]]` | `None` | CPU binding index list, e.g., `[5]` to bind to CPU core 5, index starts from 0 |

**Supported Robot Structures:**

- **`dualarm`**: Dual-arm configuration (left arm + right arm)
- **`wholebody_without_base`**: Full-body configuration without mobile base (dual-arm + torso)
- **`wholebody`**: Full-body configuration with mobile base (dual-arm + torso + base)

**Configuration Examples:**

```python
from mozrobot import MOZ1RobotConfig

# Basic dual-arm configuration
config = MOZ1RobotConfig(
    realsense_serials="123456789,987654321,555666777",
    camera_resolutions="640*480,640*480,640*480",
    structure="dualarm",
    robot_control_hz=120
)

# No camera test configuration
config_no_cam = MOZ1RobotConfig(
    no_camera=True,
    structure="dualarm"
)

# Full-body configuration
config_wholebody = MOZ1RobotConfig(
    structure="wholebody",
    robot_control_hz=120
)

# High-performance real-time configuration
config_realtime = MOZ1RobotConfig(
    structure="dualarm",
    robot_control_hz=120,
    enable_soft_realtime=True,  # Enable soft real-time scheduling
    bind_cpu_idxs=[5]  # Bind to CPU core 5
)
```

### Robot Control

#### `MOZ1Robot`

Main robot control class providing connection, control, and data collection functionality.

**Constructor:**

```python
MOZ1Robot(config: MOZ1RobotConfig)
```

**Core Methods:**

##### Connection Management

```python
connect() -> bool
```
Establish connection to robot system.
- **Return**: `True` if connection successful, `False` otherwise
- **Exception**: Raises `RuntimeError` on connection failure

```python
disconnect() -> None
```
Disconnect from robot system.

```python
is_robot_connected -> bool
```
Property to check robot connection status.
- **Return**: `True` if robot is connected, `False` otherwise
- **Usage**: Check connection status before sending commands to ensure proper robot communication

##### Robot Control

```python
enable_external_following_mode() -> None
```
Enable external following control mode.
- **Important**: Must be activated before using send_action
- **Function**: Activates robot external control receiver, making send_action commands effective
- **Exception**: Raises `RobotDeviceNotConnectedError` if robot not connected

```python
send_action(action: Dict) -> Dict
```
Send control commands to robot.
- **Parameters**:
  - `action`: Action command dictionary, data units in international units (radians, meters, seconds)
- **Return**: Confirmation of sent action command
- **Important Requirements**:
  1. **Call Frequency**: send_action calls must maintain stable frequency, consistent with `robot_control_hz` initialization parameter
  2. **Trajectory Continuity**: Sent trajectories must satisfy continuity constraints, avoiding large jumps in acceleration and jerk
  3. **Exclusive Control Mode**: Each execution structure (left arm/right arm/torso) can only use either joint space control OR Cartesian space control at any time
  4. **Prerequisite**: Must call `enable_external_following_mode()` method first
- **Exception**: Raises `RuntimeError` if robot not connected or send fails

```python
reset_robot_positions(left_arm_joints: Optional[List] = None,
                     right_arm_joints: Optional[List] = None,
                     torso_joints: Optional[List] = None,
                     gripper_positions: Optional[List] = None) -> bool
```
Asynchronously reset robot to specified positions.
- **Parameters**:
  - `left_arm_joints`: Left arm joint target positions (optional)
  - `right_arm_joints`: Right arm joint target positions (optional)
  - `torso_joints`: Torso joint target positions (optional)
  - `gripper_positions`: Gripper positions (optional)

    Data units in international units (radians, meters)
- **Return**: `True` if reset command sent successfully
- **Important**: This method is asynchronous, manual sleep required after calling to wait for robot to complete reset motion
- **Recommendation**: Wait 3-5 seconds after reset to ensure robot stabilizes, specific time depends on robot speed settings in MovaX
- **Exception**: Raises `RobotDeviceNotConnectedError` if robot not connected

##### Status Monitoring

```python
capture_observation() -> Dict
```
Capture current robot state, including images and joint data.
- **Return**: Dictionary containing observation data, data units in international units (radians, meters, seconds)
- **Camera Update Rate**: RealSense cameras update image data at 30Hz
- **Data Content**: Includes robot joint states, Cartesian positions, gripper states, and camera images
- **Rotation Representation**: All rotation data uses rotation vector representation
- **Base Control**: When robot configured as wholebody, base uses velocity control mode, returning current velocity state [vx, vy, wz]
- **Exception**: Raises `RuntimeError` if robot not connected or data capture fails

#### Additional Methods
```python
control_hz -> int
```
Check robot control frequency
- **Return**: Currently effective robot control frequency

## Data Formats

### Action Commands

Action commands are dictionaries containing various joint and end-effector control commands:

#### Gripper Commands
```python
import numpy as np

action_gripper_pos = {
    # Units: m
    "leftarm_gripper_cmd_pos": np.array([0.0]),
    "rightarm_gripper_cmd_pos": np.array([0.0]),
}
```

#### Dual-arm Commands

```python
import numpy as np

# Joint control
arm_joint_action = {
    # Left arm joint positions [joint1, joint2, ..., joint7]
    "leftarm_cmd_joint_pos": np.array([0, 0, 0, 0, 0, 0, 0]),
    # Right arm joint positions [joint1, joint2, ..., joint7]
    "rightarm_cmd_joint_pos": np.array([0, 0, 0, 0, 0, 0, 0]),
}

# Cartesian control
action_cart_action = {
    # Left arm Cartesian position [x, y, z, rx, ry, rz]
    "leftarm_cmd_cart_pos": np.array([0.3, 0.2, 0.5, 0, 0, 0]),
    # Right arm Cartesian position [x, y, z, rx, ry, rz]
    "rightarm_cmd_cart_pos": np.array([0.3, -0.2, 0.5, 0, 0, 0]),
}
```

#### Torso Commands

```python

# Joint control
torso_joint_action = {
    # Torso joint positions [joint1, joint2, ..., joint6]
    "torso_cmd_joint_pos": np.array([0, 0, 0, 0, 0, 0]),
}

# Cartesian control
torso_cart_action = {
    # Torso Cartesian position [x, y, z, rx, ry, rz]
    "torso_cmd_cart_pos": np.array([0, 0, 0, 0, 0, 0]),
}
```

#### Base Commands

**Important**: Only effective when base is in velocity control mode.

```python

# Base velocity command
cmd_vel_action = {
    # Base velocity [vx, vy, wz]
    "base_cmd_speed": np.array([0.0, 0.0, 0.0]),
}

```

### Observation Data

Observation data contains robot state information and camera images:

```python
observation = {
    # Robot state data
    "leftarm_state_cart_pos": np.array([x, y, z, rx, ry, rz]),   # Left arm Cartesian position
    "leftarm_state_joint_pos": np.array([j1, j2, ..., j7]),     # Left arm joint positions
    "rightarm_state_cart_pos": np.array([x, y, z, rx, ry, rz]), # Right arm Cartesian position
    "rightarm_state_joint_pos": np.array([j1, j2, ..., j7]),    # Right arm joint positions
    "leftarm_gripper_state_pos": np.array([gripper_pos]),       # Left gripper position
    "rightarm_gripper_state_pos": np.array([gripper_pos]),      # Right gripper position

    # Torso state (if available)
    "torso_state_cart_pos": np.array([x, y, z, rx, ry, rz]),    # Torso Cartesian position
    "torso_state_joint_pos": np.array([j1, j2, ..., j6]),       # Torso joint positions

    # Base state (if available)
    "base_state_speed": np.array([vx, vy, wz]),                 # Base velocity

    # Camera images (if enabled)
    "cam_high": np.array(shape=[H, W, 3], dtype=np.uint8),          # High camera
    "cam_left_wrist": np.array(shape=[H, W, 3], dtype=np.uint8),    # Left wrist camera
    "cam_right_wrist": np.array(shape=[H, W, 3], dtype=np.uint8),   # Right wrist camera
}
```

## Usage Examples

### Basic Usage Pattern

```python
import time
import numpy as np
from mozrobot import MOZ1Robot, MOZ1RobotConfig

# 1. Create configuration
config = MOZ1RobotConfig(
    realsense_serials="012345678,012345678,012345678",
    structure="wholebody",
    robot_control_hz=120
)

# 2. Create robot instance
robot = MOZ1Robot(config)

try:
    # 3. Connect to robot
    if robot.connect():
        print("Robot connected successfully!")

    # 4. Enable external control mode (required)
    robot.enable_external_following_mode()

    # 5. Get current state
    obs = robot.capture_observation()
    print(f"Left arm position: {obs['leftarm_state_cart_pos']}")

    # 6. Send control commands
    dt = 1.0 / robot.control_hz
    # Omit trajectory planning...
    action_list = waypoints_list()
    for action in action_list:
        action = {
            "leftarm_cmd_cart_pos": action["leftarm_action"],
            "rightarm_cmd_cart_pos": action["rightarm_action"]
        }
        robot.send_action(action)
        time.sleep(dt)

except Exception as e:
    print(f"Error: {e}")
finally:
    # 7. Disconnect
    robot.disconnect()
```

### Context Manager Pattern

```python
from mozrobot import MOZ1Robot, MOZ1RobotConfig

config = MOZ1RobotConfig(structure="dualarm")

# Automatic connection and disconnection management
with MOZ1Robot(config) as robot:
    # Enable external control mode (required)
    robot.enable_external_following_mode()

    # Get robot state
    obs = robot.capture_observation()
```

### Trajectory Control Example

```python
import time
import numpy as np
from mozrobot import MOZ1Robot, MOZ1RobotConfig

def generate_circular_trajectory(center, radius, num_points):
    """Generate circular trajectory points."""
    angles = np.linspace(0, 2*np.pi, num_points)
    trajectory = []
    for angle in angles:
        x = center[0] + radius * np.cos(angle)
        y = center[1] + radius * np.sin(angle)
        z = center[2]
        trajectory.append([x, y, z, 0, 0, 0])  # position + orientation
    return np.array(trajectory)

config = MOZ1RobotConfig(structure="dualarm")

with MOZ1Robot(config) as robot:
    # Enable external control mode (required)
    robot.enable_external_following_mode()

    # Get current position
    obs = robot.capture_observation()
    current_pos = obs["leftarm_state_cart_pos"]

    # Generate circular trajectory
    center = current_pos[:3]  # Use current position as center
    trajectory = generate_circular_trajectory(center, 0.05, 60)  # 5cm radius, 60 points

    # Execute trajectory
    dt = 1.0 / robot.control_hz
    for waypoint in trajectory:
        action = {"leftarm_cmd_cart_pos": waypoint}
        robot.send_action(action)
        time.sleep(dt)
```

## Troubleshooting

### Common Issues

#### 1. Connection Failure

**Error**: `RuntimeError: Failed to connect to robot`

**Diagnosis and Solutions**:
- Check network connectivity: `ping 172.16.0.20`
- Verify ROS_DOMAIN_ID: `echo $ROS_DOMAIN_ID`
- Ensure robot system is running
- Confirm host IP is in 172.16.0.x subnet

```bash
# Test network connectivity
ping 172.16.0.20

# Check ROS2 communication
ros2 topic list
ros2 node list
```

#### 2. Camera Initialization Failure

**Error**: Camera-related errors

**Solutions**:
- Test camera connection: `realsense-viewer`
- Use test mode: `no_camera=True`
- Verify camera serial numbers
- Ensure USB 3.0 connection

```python
# No camera test
config = MOZ1RobotConfig(
    no_camera=True,
    structure="dualarm"
)
```

#### 3. ROS2 Environment Issues

**Error**: ROS2 import errors

**Solutions**:
```bash
# Reset ROS2 environment
source /opt/ros/humble/setup.bash
export ROS_DOMAIN_ID=33

# Verify ROS2 installation
ros2 --version
```

#### 4. Python Dependencies Issues

**Error**: Module import errors

**Solutions**:
```bash
# Reinstall dependencies
pip install -r requirements.txt

# Test key dependencies
python3 -c "import torch, numpy, cv2, pyrealsense2, rclpy"
```

#### 5. Unstable ROS2 Communication

**Error**: Unstable ROS2 communication due to network issues, causing robot jittering during action execution.

**Problem Detection**:
1. `ros2 topic hz /cart_states` shows unstable ROS2 frequency with abnormal fluctuations.

**Solutions**:

**Note**: The configuration file provided below assumes your IP address is 172.16.0.30. Please modify it according to your actual situation.

Add FastDDS configuration file `/path/to/fastdds_config.xml`:
```xml
<?xml version="1.0" encoding="UTF-8" ?>
<dds xmlns="http://www.eprosima.com">
    <profiles>
        <transport_descriptors>
            <transport_descriptor>
                <transport_id>udp_transport</transport_id>
                <type>UDPv4</type>
                <interfaceWhiteList>
                    <address>172.16.0.20</address>
                    <address>172.16.0.30</address>
                    <address>127.0.0.1</address>
                </interfaceWhiteList>
            </transport_descriptor>
        </transport_descriptors>

        <participant profile_name="UDPParticipant" is_default_profile="true">
            <rtps>
                <userTransports>
                    <transport_id>udp_transport</transport_id>
                </userTransports>
                <useBuiltinTransports>false</useBuiltinTransports>
            </rtps>
        </participant>
    </profiles>
</dds>
```

Execute before running programs:
```bash
export FASTRTPS_DEFAULT_PROFILES_FILE=/path/to/fastdds_config.xml
```

#### 6. Python Interpreter and ROS2 Compatibility Issues

**Error**: Python module import errors, rclpy initialization failure, or version conflicts

**Problem Description**:
ROS2 has compatibility issues with Python virtual environments (conda, venv), mainly due to:
- ROS2 binaries bound to system default Python version
- conda environment modifying default Python interpreter path
- Python package path conflicts preventing proper import of rclpy and other ROS2 modules

**Solutions**:

##### Option 1: Use System Python (Recommended)
```bash
# Confirm using system Python
which python3
# Output should be: /usr/bin/python3

# Install Python dependencies to system environment
sudo apt install python3-pip
pip3 install --user [required_package_name]

# Avoid activating conda base environment
conda config --set auto_activate_base false
```

##### Option 2: Conda Environment Isolation Configuration
```bash
# Configure environment variable order correctly in ~/.bashrc
# Load ROS2 environment first, then initialize conda
source /opt/ros/humble/setup.bash
export ROS_DOMAIN_ID=33
# conda initialize block should be at the end

# Create dedicated conda environment (matching system Python version)
conda create -n ros2_mozrobot python=3.x  # 3.x is system Python version
conda activate ros2_mozrobot

```

**Best Practices**:
1. **Prioritize using system Python environment** for ROS2 development to avoid complex compatibility issues
2. **If conda must be used**, ensure Python version matches system version

**Reference Resources**:
- [ROS2 Official Python Package Usage Guide](https://docs.ros.org/en/humble/How-To-Guides/Using-Python-Packages.html)

### Debugging and Monitoring

#### Network Diagnostics

```bash
# Test robot connectivity
ping 172.16.0.20

# Check ROS2 communication
ros2 topic list
ros2 node list

# Monitor robot data
ros2 topic echo /cart_states

# Monitor robot commands
ros2 topic echo /mx_mix_command
```

---

**Happy robot programming! 🤖**