# MozRobot SDK 使用文档

[![Python 3.10+](https://img.shields.io/badge/python-3.10+-blue.svg)](https://www.python.org/downloads/)
[![ROS2 Humble](https://img.shields.io/badge/ROS2-Humble-brightgreen.svg)](https://docs.ros.org/en/humble/)

## 概述

MozRobot SDK 是一个用于控制 Moz 机器人系统的综合性 Python SDK。基于 ROS2 构建，它为Moz1机器人提供直观的机器人控制、数据采集的API。

### 主要特性

- **多配置支持**: 双臂、全身（有/无底盘）
- **实时控制**: 高频控制循环（最高 120Hz）实现精确操控
- **视觉集成**: 原生 RealSense 相机支持，支持多相机配置
- **ROS2 原生**: 与 ROS2 Humble 生态系统无缝集成
- **安全优先**: 内置安全检查和错误处理
- **易于集成**: 简单的 Python API，便于快速原型开发和部署

## 快速开始

### 性能优化配置（可选）

MozRobot SDK 支持实时性能优化，适用于对控制精度和延迟有严格要求的场景：

#### 软实时调度
启用软实时调度可以提高机器人控制的实时性能，减少控制延迟和抖动。

```bash
# 执行设置脚本（需要管理员权限）
sudo bash scripts/setup_rtprio.sh
sudo reboot
```

#### CPU绑定
将机器人控制线程绑定到特定CPU核心，避免任务切换开销，提高控制稳定性。

**建议配置**：
- 避免使用核心0，通常被系统任务占用

**重要注意事项**：
- 这些功能是可选的，普通使用不需要配置
- 仅在需要最佳实时性能时启用
- 确保系统有足够的CPU资源用于绑定

### 系统要求

- **操作系统**: Ubuntu 22.04 LTS (推荐)
- **ROS 版本**: ROS2 Humble
- **Python 版本**: 3.10 或更高版本，建议使用系统解释器
- **硬件要求**: 
  - RealSense 相机：USB 3.0 * 3
  - 与机器人控制系统连接：RJ45 * 1
- **系统资源**: 推荐 8GB+ 内存

### 支持平台

- x86_64
- aarch64

### 网络配置

- **主机 IP**: 必须在 172.16.0.x 子网中（例如：172.16.0.100）
- **机器人 IP**: 172.16.0.20
- **ROS_DOMAIN_ID**: 33（需要与 MovaX 控制系统保持一致）

## 环境配置参考


### 步骤 1：系统依赖

[ros-humble安装指南](https://docs.ros.org/en/humble/Installation/Ubuntu-Install-Debs.html)

```bash
# 安装 ROS2 Humble
sudo apt install software-properties-common
sudo add-apt-repository universe
sudo apt update && sudo apt install curl -y

sudo curl -sSL https://raw.githubusercontent.com/ros/rosdistro/master/ros.key -o /usr/share/keyrings/ros-archive-keyring.gpg
echo "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/ros-archive-keyring.gpg] http://packages.ros.org/ros2/ubuntu $(. /etc/os-release && echo $UBUNTU_CODENAME) main" | sudo tee /etc/apt/sources.list.d/ros2.list > /dev/null

sudo apt update
# you can install ros-humble-desktop-full
sudo apt install -y ros-humble-ros-base python3-colcon-common-extensions ros-dev-tools

# 安装系统依赖
sudo apt install -y build-essential cmake git curl wget python3-pip python3-dev
sudo apt install -y libopencv-dev python3-opencv
```

### 步骤 2：robot_interface ROS包构建

robot_interface 是必需的ROS2接口包，包含机器人通信消息和服务定义。必须先构建此包才能正常使用SDK。

```bash
# 进入依赖包目录
cd /path/to/MozRobot-SDK/deps

# 创建ROS2工作空间并构建robot_interface包
mkdir -p ~/ros_ws/src
cp -r robot_interface ~/ros_ws/src/
cd ~/ros_ws

# 配置ROS2环境
source /opt/ros/humble/setup.bash
# 构建包
colcon build --packages-select mc_core_interface
```

### 步骤 3：环境配置

```bash
# 配置 ROS2 环境
echo "source /opt/ros/humble/setup.bash" >> ~/.bashrc
echo "export ROS_DOMAIN_ID=33" >> ~/.bashrc
echo "source ~/ros_ws/install/setup.bash" >> ~/.bashrc  # 添加robot_interface包环境
source ~/.bashrc
```

### 步骤 4：网络配置

编辑网络配置：

```bash
sudo nano /etc/netplan/01-netcfg.yaml
```

添加以下配置：

```yaml
network:
  version: 2
  ethernets:
    eth0:  # 替换为实际网络接口名称
      addresses: [172.16.0.100/24]  # 选择 172.16.0.x 范围内的 IP
      gateway4: 172.16.0.1
      nameservers:
        addresses: [8.8.8.8, 8.8.4.4]
```

应用配置：

```bash
sudo netplan apply
```

#### 步骤 5：SDK 安装

```bash
# 安装 Python 依赖
cd /path/to/MozRobot-sdk
pip3 install -r packages/requirements.txt

# 安装sdk
pip install packages/MozRobot-*.whl
```

#### 步骤 6：验证安装

```bash
python3 -c "from MozRobot import MOZ1Robot, MOZ1RobotConfig; print('✅ MozRobot SDK 安装成功!')"
```

### 快速测试

```bash
# 设置环境
export ROS_DOMAIN_ID=33

# 运行示例
cd MozRobot
python3 example/example_usage.py

# 启用性能优化的示例
python3 example/example_usage.py --enable-soft-realtime --bind-cpu 5 --no-camera
```

## API 参考

### 机器人配置

#### `MOZ1RobotConfig`

用于设置机器人参数的主要配置类。

**构造函数参数：**

| 参数名 | 类型 | 默认值 | 描述 |
|--------|------|--------|------|
| `realsense_serials` | `str` | `"0000000000,0000000000,0000000000"` | RealSense 相机序列号（逗号分隔） |
| `camera_resolutions` | `str` | `"320*240,320*240,320*240"` | 相机分辨率，格式为"宽*高" |
| `no_camera` | `bool` | `False` | 跳过相机初始化（用于测试） |
| `structure` | `str` | `"wholebody"` | 机器人配置类型 |
| `robot_control_hz` | `int` | `120` | 控制循环频率（Hz），与机器人控制系统中的设定保持一致 |
| `enable_soft_realtime` | `bool` | `False` | 启用软实时调度，需要先执行 `scripts/setup_rtprio.sh` |
| `bind_cpu_idxs` | `Optional[Sequence[int]]` | `None` | CPU绑定索引列表，如 `[5]` 表示绑定到第5个CPU核心，索引从0开始 |

**支持的机器人结构：**

- **`dualarm`**: 双臂配置（左臂 + 右臂）
- **`wholebody_without_base`**: 无移动底盘的全身配置（双臂 + 躯干）
- **`wholebody`**: 带移动底盘的全身配置（双臂 + 躯干 + 底盘）

**配置示例：**

```python
from MozRobot import MOZ1RobotConfig

# 基础双臂配置
config = MOZ1RobotConfig(
    realsense_serials="123456789,987654321,555666777",
    camera_resolutions="640*480,640*480,640*480",
    structure="dualarm",
    robot_control_hz=120
)

# 无相机测试配置
config_no_cam = MOZ1RobotConfig(
    no_camera=True,
    structure="dualarm"
)

# 全身配置
config_wholebody = MOZ1RobotConfig(
    structure="wholebody",
    robot_control_hz=120
)

# 高性能实时配置
config_realtime = MOZ1RobotConfig(
    structure="dualarm",
    robot_control_hz=120,
    enable_soft_realtime=True,  # 启用软实时调度
    bind_cpu_idxs=[5]  # 绑定到CPU核心5
)
```

### 机器人控制

#### `MOZ1Robot`

提供连接、控制和数据采集功能的主要机器人控制类。

**构造函数：**

```python
MOZ1Robot(config: MOZ1RobotConfig)
```

**核心方法：**

##### 连接管理

```python
connect() -> bool
```
建立与机器人系统的连接。
- **返回值**: 连接成功返回 `True`，否则返回 `False`
- **异常**: 连接失败时抛出 `RuntimeError`

```python
disconnect() -> None
```
断开与机器人系统的连接。

```python
is_robot_connected -> bool
```
检查机器人连接状态的属性。
- **返回值**: 机器人已连接返回 `True`，否则返回 `False`
- **用法**: 在发送指令前检查连接状态，确保机器人通信正常

##### 机器人控制

```python
enable_external_following_mode() -> None
```
启用外部跟随控制模式。
- **重要**: 必须使用send_action前激活外部跟随模式
- **功能**: 激活机器人外部控制接收器，使send_action发送的指令生效
- **异常**: 机器人未连接时抛出 `RobotDeviceNotConnectedError`

```python
send_action(action: Dict) -> Dict
```
向机器人发送控制命令。
- **参数**:
  - `action`: 动作命令字典，数据单位为国际单位（弧度，米，秒）
- **返回值**: 发送的动作命令确认
- **重要要求**:
  1. **调用频率**: send_action的调用必须保持稳定频率，与`robot_control_hz`初始化参数一致
  2. **轨迹连续性**: 发送的轨迹必须满足连续性约束，避免加速度、加加速度出现大幅突变
  3. **控制模式独占**: 每个执行结构（左臂/右臂/躯干）同一时间只能选择轴空间控制或笛卡尔空间控制其中一种模式
  4. **前置条件**: 必须先调用`enable_external_following_mode()`方法
- **异常**: 机器人未连接或发送失败时抛出 `RuntimeError`

```python
reset_robot_positions(left_arm_joints: Optional[List] = None,
                     right_arm_joints: Optional[List] = None,
                     torso_joints: Optional[List] = None,
                     gripper_positions: Optional[List] = None) -> bool
```
异步复位机器人到指定位置。
- **参数**:
  - `left_arm_joints`: 左臂关节目标位置（可选）
  - `right_arm_joints`: 右臂关节目标位置（可选）
  - `torso_joints`: 躯干关节目标位置（可选）
  - `gripper_positions`: 夹爪位置（可选）

    数据单位为国际单位（弧度，米）
- **返回值**: 复位指令发送成功返回 `True`
- **重要**: 此方法是异步的，调用后需要手动sleep等待机器人完成复位动作
- **建议**: 复位后等待3-5秒确保机器人稳定到位，具体时间取决于MovaX中设置的机器人速度
- **异常**: 机器人未连接时抛出 `RobotDeviceNotConnectedError`

##### 状态监控

```python
capture_observation() -> Dict
```
捕获当前机器人状态，包括图像和关节数据。
- **返回值**: 包含观测数据的字典，数据单位为国际单位（弧度，米，秒）
- **相机更新频率**: RealSense相机以30Hz频率更新图像数据
- **数据内容**: 包含机器人关节状态、笛卡尔位置、夹爪状态和相机图像
- **旋转表示**: 所有旋转数据使用旋转向量（rotation vector）表示法
- **底盘控制**: 当机器人配置为wholebody时，底盘采用速度控制模式，返回当前速度状态 [vx, vy, wz]
- **异常**: 机器人未连接或数据捕获失败时抛出 `RuntimeError`

#### 更多方法
```python
control_hz -> int
```
检查机器人的控制频率
- **返回值**: 当前生效的机器人控制频率

##### 机器人控制

## 数据格式

### 动作命令

动作命令是包含各种关节和末端执行器控制命令的字典：

#### 夹爪指令
```python
import numpy as np

action_gripper_pos = {
    # 单位：m
    "leftarm_gripper_cmd_pos": np.array([0.0]),
    "rightarm_gripper_cmd_pos": np.array([0.0]),
}
```

#### 双臂命令

```python
import numpy as np

# 关节控制
arm_joint_action = {
    # 左臂关节位置 [joint1, joint2, ..., joint7]
    "leftarm_cmd_joint_pos": np.array([0, 0, 0, 0, 0, 0, 0]),
    # 右臂关节位置 [joint1, joint2, ..., joint7]
    "rightarm_cmd_joint_pos": np.array([0, 0, 0, 0, 0, 0, 0]),
}

# 笛卡尔控制
action_cart_action = {
    # 左臂笛卡尔位置 [x, y, z, rx, ry, rz]
    "leftarm_cmd_cart_pos": np.array([0.3, 0.2, 0.5, 0, 0, 0]),
    # 右臂笛卡尔位置 [x, y, z, rx, ry, rz]
    "rightarm_cmd_cart_pos": np.array([0.3, -0.2, 0.5, 0, 0, 0]),
}
```

#### 躯干指令

```python

# 关节控制
torso_joint_action = {
    # 躯干关节位置 [joint1, joint2, ..., joint6]
    "torso_cmd_joint_pos": np.array([0, 0, 0, 0, 0, 0]),
}

# 笛卡尔控制
torso_cart_action = {
    # 躯干笛卡尔位置 [x, y, z, rx, ry, rz]
    "torso_cmd_cart_pos": np.array([0, 0, 0, 0, 0, 0]),
}
```

#### 底盘指令

**重要**：仅在底盘处于速度控制模式下生效。

```python

# 底盘速度指令
cmd_vel_action = {
    # 躯干关节位置 [vx, vy, wz]
    "base_cmd_speed": np.array([0.0, 0.0, 0.0]),
}

```

### 观测数据

观测数据包含机器人状态信息和相机图像：

```python
observation = {
    # 机器人状态数据
    "leftarm_state_cart_pos": np.array([x, y, z, rx, ry, rz]),   # 左臂笛卡尔位置
    "leftarm_state_joint_pos": np.array([j1, j2, ..., j7]),     # 左臂关节位置
    "rightarm_state_cart_pos": np.array([x, y, z, rx, ry, rz]), # 右臂笛卡尔位置
    "rightarm_state_joint_pos": np.array([j1, j2, ..., j7]),    # 右臂关节位置
    "leftarm_gripper_state_pos": np.array([gripper_pos]),       # 左夹爪位置
    "rightarm_gripper_state_pos": np.array([gripper_pos]),      # 右夹爪位置

    # 躯干状态（如果可用）
    "torso_state_cart_pos": np.array([x, y, z, rx, ry, rz]),    # 躯干笛卡尔位置
    "torso_state_joint_pos": np.array([j1, j2, ..., j6]),       # 躯干关节位置

    # 底盘状态（如果可用）
    "base_state_speed": np.array([vx, vy, wz]),                 # 底盘速度

    # 相机图像（如果启用）
    "cam_high": np.array(shape=[H, W, 3], dtype=np.uint8),          # 高位相机
    "cam_left_wrist": np.array(shape=[H, W, 3], dtype=np.uint8),    # 左手腕相机
    "cam_right_wrist": np.array(shape=[H, W, 3], dtype=np.uint8),   # 右手腕相机
}
```

## 使用示例

### 基本使用模式

```python
import time
import numpy as np
from MozRobot import MOZ1Robot, MOZ1RobotConfig

# 1. 创建配置
config = MOZ1RobotConfig(
    realsense_serials="012345678,012345678,012345678",
    structure="wholebody",
    robot_control_hz=120
)

# 2. 创建机器人实例
robot = MOZ1Robot(config)

try:
    # 3. 连接机器人
    if robot.connect():
        print("机器人连接成功！")

    # 4. 启用外部控制模式（必需）
    robot.enable_external_following_mode()

    # 5. 获取当前状态
    obs = robot.capture_observation()
    print(f"左臂位置: {obs['leftarm_state_cart_pos']}")

    # 6. 发送控制命令
    dt = 1.0 / robot.control_hz
    # 省略轨迹规划...
    action_list = waypoints_list()
    for action in action_list
        action = {
            "leftarm_cmd_cart_pos": action["leftarm_action"],
            "rightarm_cmd_cart_pos": action["rightarm_action"]
        }
        robot.send_action(action)
        time.sleep(dt)

except Exception as e:
    print(f"错误: {e}")
finally:
    # 7. 断开连接
    robot.disconnect()
```

### 上下文管理器模式

```python
from MozRobot import MOZ1Robot, MOZ1RobotConfig

config = MOZ1RobotConfig(structure="dualarm")

# 自动连接和断开管理
with MOZ1Robot(config) as robot:
    # 启用外部控制模式（必需）
    robot.enable_external_following_mode()

    # 获取机器人状态
    obs = robot.capture_observation()
```

### 轨迹控制示例

```python
import time
import numpy as np
from MozRobot import MOZ1Robot, MOZ1RobotConfig

def generate_circular_trajectory(center, radius, num_points):
    """生成圆形轨迹点。"""
    angles = np.linspace(0, 2*np.pi, num_points)
    trajectory = []
    for angle in angles:
        x = center[0] + radius * np.cos(angle)
        y = center[1] + radius * np.sin(angle)
        z = center[2]
        trajectory.append([x, y, z, 0, 0, 0])  # 位置 + 姿态
    return np.array(trajectory)

config = MOZ1RobotConfig(structure="dualarm")

with MOZ1Robot(config) as robot:
    # 启用外部控制模式（必需）
    robot.enable_external_following_mode()

    # 获取当前位置
    obs = robot.capture_observation()
    current_pos = obs["leftarm_state_cart_pos"]

    # 生成圆形轨迹
    center = current_pos[:3]  # 使用当前位置作为中心
    trajectory = generate_circular_trajectory(center, 0.05, 60)  # 5cm 半径，60 个点

    # 执行轨迹
    dt = 1.0 / robot.control_hz
    for waypoint in trajectory:
        action = {"leftarm_cmd_cart_pos": waypoint}
        robot.send_action(action)
        time.sleep(dt)
```


## 故障排除

### 常见问题

#### 1. 连接失败

**错误**: `RuntimeError: 连接机器人失败`

**诊断和解决方案**：
- 检查网络连通性：`ping 172.16.0.20`
- 验证 ROS_DOMAIN_ID：`echo $ROS_DOMAIN_ID`
- 确保机器人系统正在运行
- 确认主机 IP 在 172.16.0.x 子网中

```bash
# 测试网络连通性
ping 172.16.0.20

# 检查 ROS2 通信
ros2 topic list
ros2 node list
```

#### 2. 相机初始化失败

**错误**: 相机相关错误

**解决方案**：
- 测试相机连接：`realsense-viewer`
- 使用测试模式：`no_camera=True`
- 验证相机序列号
- 确保 USB 3.0 连接

```python
# 无相机测试
config = MOZ1RobotConfig(
    no_camera=True,
    structure="dualarm"
)
```

#### 3. ROS2 环境问题

**错误**: ROS2 导入错误

**解决方案**：
```bash
# 重置 ROS2 环境
source /opt/ros/humble/setup.bash
export ROS_DOMAIN_ID=33

# 验证 ROS2 安装
ros2 --version
```

#### 4. Python 依赖问题

**错误**: 模块导入错误

**解决方案**：
```bash
# 重新安装依赖
pip install -r requirements.txt

# 测试关键依赖
python3 -c "import torch, numpy, cv2, pyrealsense2, rclpy"
```

#### 5. ROS2通讯不稳定

**错误**：由于网络原因造成ROS2通讯不稳定，导致机器人执行action时抖动。

**问题判别**：
1. `ros2 topic hz /cart_states`观测到ros2效率的频率不稳定，存在异常波动。

**解决方案**：

**注意**：下文提供的配置文件假设你的IP地址为172.16.0.30，请根据你的实际情况修改。

新增fastdds配置文件 `/path/to/fastdds_config.xml`：
```
<?xml version="1.0" encoding="UTF-8" ?>
<dds xmlns="http://www.eprosima.com">
    <profiles>
        <transport_descriptors>
            <transport_descriptor>
                <transport_id>udp_transport</transport_id>
                <type>UDPv4</type>
                <interfaceWhiteList>
                    <address>172.16.0.20</address>
                    <address>172.16.0.30</address>
                    <address>127.0.0.1</address>
                </interfaceWhiteList>
            </transport_descriptor>
        </transport_descriptors>

        <participant profile_name="UDPParticipant" is_default_profile="true">
            <rtps>
                <userTransports>
                    <transport_id>udp_transport</transport_id>
                </userTransports>
                <useBuiltinTransports>false</useBuiltinTransports>
            </rtps>
        </participant>
    </profiles>
</dds
```

每次执行程序前执行：
```
export FASTRTPS_DEFAULT_PROFILES_FILE=/path/to/fastdds_config.xml
```


#### 6. Python解释器与ROS2兼容性问题

**错误**: Python模块导入错误、rclpy初始化失败或版本冲突

**问题描述**：
ROS2与Python虚拟环境（如conda、venv）存在兼容性问题，主要原因包括：
- ROS2二进制文件与系统默认Python版本绑定
- conda环境修改了默认Python解释器路径
- Python包路径冲突导致rclpy等ROS2模块无法正确导入

**解决方案**：

##### 方案1：使用系统Python（推荐）
```bash
# 确认使用系统Python
which python3
# 输出应该是：/usr/bin/python3

# 安装Python依赖到系统环境
sudo apt install python3-pip
pip3 install --user [所需包名]

# 避免激活conda base环境
conda config --set auto_activate_base false
```

##### 方案2：conda环境隔离配置
```bash
# 在 ~/.bashrc 中正确配置环境变量顺序
# 先加载ROS2环境，再初始化conda
source /opt/ros/humble/setup.bash
export ROS_DOMAIN_ID=33
# conda initialize block should be at the end

# 创建专用conda环境（与系统Python版本匹配）
conda create -n ros2_MozRobot python=3.x  # 3.x为系统Python版本
conda activate ros2_MozRobot


```

**最佳实践**：
1. **优先使用系统Python环境**进行ROS2开发，避免复杂的兼容性问题
2. **如必须使用conda**，确保Python版本与系统一致

**参考资源**：
- [ROS2官方Python包使用指南](https://docs.ros.org/en/humble/How-To-Guides/Using-Python-Packages.html)

### 调试和监控

#### 网络诊断

```bash
# 测试机器人连通性
ping 172.16.0.20

# 检查 ROS2 通信
ros2 topic list
ros2 node list

# 监控机器人数据
ros2 topic echo /cart_states

# 监控机器人指令
ros2 topic echo /mx_mix_command
```


---

**祝您机器人编程愉快！🤖**